<?php

namespace Drupal\isimo\Controller;

use Drupal;
use Drupal\Core\Controller\ControllerBase;
use Exception;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Yaml\Yaml;

/**
 * Handles requests for isimo status.
 */
class StatusController extends ControllerBase
{

  public function main($accesskey = null)
  {
    // Validate access key
    $key = \Drupal::config('isimo.settings')->get('accesskey');
    if (empty($accesskey) || empty($key) || $accesskey !== $key) {
      // Not valid, exit
      throw new NotFoundHttpException();
    }

    // Invalidate data for all projects, attempt to re-fetch, and
    // trigger any configured notifications about the new status.
    update_refresh();
    update_fetch_data();

    // Init data array
    $data = [
      'software' => 'Drupal',
      'version' => Drupal::VERSION,
    ];

    try {
      // Get status report
      module_load_include('inc', 'update', 'update.report');
      $available = update_get_available(true);
      $data['report'] = update_calculate_project_data($available);
    } catch (Exception $e) {
      $data['report'] = 'ERROR:' . $e->getMessage();
    }

    try {
      // Get phpinfo
      ob_start();
      phpinfo();
      $data['phpinfo'] = ob_get_clean();
    } catch (Exception $e) {
      $data['phpinfo'] = 'ERROR:' . $e->getMessage();
    }

    try {
      // Get MySQL info
      $result = Drupal::database()->query('SHOW VARIABLES');
      $data['mysql'] = [];
      while ($row = $result->fetchAssoc()) {
        $data['mysql'][$row['Variable_name']] = $row['Value'];
      }
    } catch (Exception $e) {
      $data['mysql'] = 'ERROR:' . $e->getMessage();
    }

    // Include client version
    try {
      $yml = file_get_contents(__DIR__ . '/../../isimo.info.yml');
      $yml = Yaml::parse($yml);
      if (isset($yml['version'])) {
        $data['client_version'] = $yml['version'];
      }
    } catch (Exception $e) {
      $data['client_version'] = 'ERROR:' . $e->getMessage();
    }

    // Include composer info
    $composer_path = DRUPAL_ROOT . '/composer.lock';
    if (is_file($composer_path)) {
      // Get composer lock file
      try {
        $composer_lock = file_get_contents($composer_path);
        if ($composer_lock) {
          $data['composer_lock'] = $composer_lock;
        }
      } catch (Exception $e) {
        $data['composer_lock'] = 'ERROR:' . $e->getMessage();
      }
      // Include composer outdated and diagnose info if possible
      try {
        $bins = [
          '/usr/bin/composer',
          '/usr/local/bin/composer',
          DRUPAL_ROOT . '/composer.phar',
          DRUPAL_ROOT . '/scripts/composer.phar',
        ];
        foreach ($bins as $bin) {
          if (file_exists($bin)) {
            $cmd_parts = [];
            $cmd_parts[] = 'COMPOSER_HOME=' . escapeshellarg(sys_get_temp_dir());
            $cmd_parts[] = escapeshellcmd($bin);
            $cmd_parts[] = '--working-dir=' . escapeshellarg(DRUPAL_ROOT);

            $cmd_parts['composer_command'] = escapeshellarg('outdated');
            $cmd_parts[] = ' 2>&1';
            $data['composer_outdated'] = shell_exec(implode(' ', $cmd_parts));

            $cmd_parts['composer_command'] = escapeshellarg('diagnose');
            $data['composer_diagnose'] = shell_exec(implode(' ', $cmd_parts));

            break;
          }
        }
      } catch (Exception $e) {
        $data['composer_outdated'] = 'ERROR:' . $e->getMessage();
        $data['composer_diagnose'] = 'ERROR:' . $e->getMessage();
      }
    }

    // Output data
    $out = json_encode($data);
    if ($out) {
      header("Content-type: application/json");
      echo $out;
    } else {
      echo serialize($data);
    }
    exit;

  }

}
