<?php
/**
 * Copyright © 2017 Spiro Kommunikation AB. All rights reserved.
 */

namespace Spiro\UrlKeyMask\Plugin;

class ProductPlugin
{

    const XML_PATH_AUTO_GENERATE_MASK = 'catalog/fields_masks/url_key';

    /**
     * @var string $baseKey
     */
    protected $baseKey;
    /**
     * @var string $urlKey
     */
    protected $urlKey;
    /**
     * @var \Magento\Catalog\Model\Product $product
     */
    protected $product;
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    protected $productRepository;
    /**
     * @var \Magento\Framework\Api\SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     *
     * @internal param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    ) {
        $this->productRepository     = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->scopeConfig           = $scopeConfig;
    }

    /**
     * Generate product url key based on url_key entered by merchant or product name
     *
     * @param \Magento\CatalogUrlRewrite\Model\ProductUrlPathGenerator $subject
     * @param \Magento\Catalog\Model\Product $product
     */
    public function beforeGetUrlKey(
        \Magento\CatalogUrlRewrite\Model\ProductUrlPathGenerator $subject,
        \Magento\Catalog\Model\Product $product
    ) {
        // Do not set product if it already has a url key (essentialy a "disable behaviour" feature)
        if (empty($product->getUrlKey())) {
            // Clear generated key if not the same product as before
            if ($product !== $this->product) {
                $this->urlKey = null;
            }
            $this->product = $product;
        }
    }

    /**
     * Generate product url key based on url_key entered by merchant or product name
     *
     * @param \Magento\CatalogUrlRewrite\Model\ProductUrlPathGenerator $subject
     * @param string $return
     *
     * @return string
     */
    public function afterGetUrlKey(
        \Magento\CatalogUrlRewrite\Model\ProductUrlPathGenerator $subject,
        $return
    ) {
        // Return parent response if product was not set or parent response was bool false
        if (! $this->product || $return === false) {
            return $return;
        }

        // Return already generated url key or generate it now
        return $this->urlKey ? : $this->generateUniqueUrlKey();
    }

    /**
     * Generate unique product url key
     *
     * @return string
     */
    protected function generateUniqueUrlKey()
    {
        // Generate url key
        $this->generateUrlKey();
        if (! $this->urlKeyExists()) {
            return $this->urlKey;
        }

        // Generate url key with numerical suffix
        $n = 1;
        $this->generateUrlKey($n++);
        while ($this->urlKeyExists()) {
            $this->generateUrlKey($n++);
        }

        return $this->urlKey;
    }

    /**
     * Generate url key
     *
     * @param null|string $suffix
     */
    protected function generateUrlKey($suffix = null)
    {
        // Generate base key if url key has not been generated
        if (! $this->urlKey) {
            // Get configured mask
            $mask = $this->scopeConfig->getValue(ProductPlugin::XML_PATH_AUTO_GENERATE_MASK);
            // Replace placeholders
            $replace = [
                '{{name}}' => $this->product->getName(),
                '{{sku}}'  => $this->product->getSku(),
            ];
            $mask    = str_ireplace(array_keys($replace), array_values($replace), $mask);
            // Replace date placeholder(s)
            $this->baseKey = preg_replace_callback("/{{date([^}]*)}}/isU", function ($m) {
                // Make sure pattern is set
                if (! isset($m[ 1 ])) {
                    $m[ 1 ] = ':';
                }
                // Remove initial colon from pattern
                $m[ 1 ] = substr($m[ 1 ], 1);
                if (empty($m[ 1 ])) {
                    // Fall back to default format
                    $m[ 1 ] = 'Y-m-d';
                }

                // Return formatted date
                return date($m[ 1 ]);
            }, $mask);
        }

        // Add suffix counter if provided
        $key = $this->baseKey . $suffix ? : '';

        // Generate url key
        $this->urlKey = $this->product->formatUrlKey($key);
    }

    /**
     * Check if generated url key exists
     *
     * @return bool
     */
    protected function urlKeyExists()
    {
        $criteria = $this->searchCriteriaBuilder
            ->addFilter('url_key', $this->urlKey, 'eq')
            ->setPageSize(1)
            ->create();

        return $this->productRepository->getList($criteria)->getTotalCount() > 0;
    }

}
