<?php

namespace Drupal\spiro_utils\Services;

use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Node helper class
 */
class Http
{

  /**
   * Read JSON from HTTP response and return array
   * @param string $url
   * @return array|null
   */
  public static function getJSON($url)
  {
    // Get response body
    $reponse = self::get($url);
    if (!$reponse) {
      \Drupal::logger('spiro_utils_http')->error('No reponse for ' . $url);
      return null;
    }

    // Remove HTML comments that may be added to the reponse
    $reponse = preg_replace("/<!--(.*)-->/", '', $reponse);

    // Decode and return response
    return json_decode($reponse);
  }

  /**
   * Read and return HTTP response through Guzzle
   * @param string $url
   * @return string
   */
  public static function get($url)
  {
    // Get Guzzle client
    $client = \Drupal::httpClient();
    try {
      $response = $client->request('GET', $url);
    } catch (GuzzleException $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    } catch (Exception $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    }

    // Check status
    if ($response->getStatusCode() !== 200) {
      \Drupal::logger('spiro_utils_http')->error($url . ' returned status ' . $response->getStatusCode());
    }

    // Return response body
    return (string)$response->getBody();
  }

  /**
   * Redirect (exists application)
   * @param string|NodeInterface $url string or node
   * @param array $parameters Optional (used when $url is a route)
   */
  public static function redirect($url, $parameters = [])
  {
    $response = new RedirectResponse(self::url($url, $parameters));
    $response->send();
  }

  /**
   * Render URL
   * @param string|NodeInterface $url string or node or route
   * @param array $parameters Optional (used when $url is a route)
   * @return string
   */
  public static function url($url, $parameters = [])
  {
    try {
      if ($url instanceof NodeInterface) {
        $url = $url->toUrl()->toString();
      } else if (strpos($url, '.')) {
        $url = Url::fromRoute($url, $parameters)->toString() ?: $url;
      }
    } catch (EntityMalformedException $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    } catch (Exception $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    }
    $url = (string)$url;
    if (strpos($url, '://') === false) {
      $url = \Drupal::request()->getSchemeAndHttpHost() . $url;
    }
    return $url;
  }

}
