<?php

namespace Drupal\spiro_utils\Services;

use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Node helper class
 */
class Http
{

  /**
   * Read JSON from HTTP response and return array
   * @param string $url
   * @return array|null
   */
  public static function getJSON($url)
  {
    // Get response body
    $reponse = self::get($url);
    if (!$reponse) {
      \Drupal::logger('spiro_utils_http')->error('No reponse for ' . $url);
      return null;
    }

    // Remove HTML comments that may be added to the reponse
    $reponse = preg_replace("/<!--(.*)-->/", '', $reponse);

    // Decode and return response
    return json_decode($reponse);
  }

  /**
   * Read and return HTTP response through Guzzle
   * @param string $url
   * @return string
   */
  public static function get($url)
  {
    // Get Guzzle client
    $client = \Drupal::httpClient();
    try {
      $response = $client->request('GET', $url);
    } catch (GuzzleException $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    } catch (Exception $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    }

    // Check status
    if ($response->getStatusCode() !== 200) {
      \Drupal::logger('spiro_utils_http')->error($url . ' returned status ' . $response->getStatusCode());
    }

    // Return response body
    return (string)$response->getBody();
  }

  /**
   * Redirect (exists application)
   * @param string|NodeInterface $url string or node
   * @param array $parameters Optional (used when $url is a route)
   */
  public static function redirect($url, $parameters = [])
  {
    $response = new RedirectResponse(self::url($url, $parameters));
    $response->send();
  }

  /**
   * Render URL
   * @param string|NodeInterface $url string or node or route
   * @param array $parameters Optional (used when $url is a route)
   * @return string
   */
  public static function url($url, $parameters = [])
  {
    try {
      if ($url instanceof NodeInterface) {
        $url = $url->toUrl()->toString();
      } else if (strpos($url, '.')) {
        $url = Url::fromRoute($url, $parameters)->toString() ?: $url;
      }
    } catch (EntityMalformedException $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    } catch (Exception $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    }
    $url = (string)$url;
    if (strpos($url, '://') === false) {
      $url = \Drupal::request()->getSchemeAndHttpHost() . $url;
    }
    return $url;
  }

  /**
   * Send status code as response along with a message
   * @param int $code
   * @param null|string $message If message is omitted the standard message is used
   */
  public function sendStatus($code, $message = null)
  {
    $defaut_messages = [
      100 => 'Continue',
      101 => 'Switching Protocols',
      102 => 'Processing (WebDAV; RFC 2518)',
      103 => 'Early Hints (RFC 8297)',
      200 => 'OK',
      201 => 'Created',
      202 => 'Accepted',
      203 => 'Non-Authoritative Information (since HTTP/1.1)',
      204 => 'No Content',
      205 => 'Reset Content',
      206 => 'Partial Content (RFC 7233)',
      207 => 'Multi-Status (WebDAV; RFC 4918)',
      208 => 'Already Reported (WebDAV; RFC 5842)',
      226 => 'IM Used (RFC 3229)',
      300 => 'Multiple Choices',
      301 => 'Moved Permanently',
      302 => 'Found',
      303 => 'See Other (since HTTP/1.1)',
      304 => 'Not Modified (RFC 7232)',
      305 => 'Use Proxy (since HTTP/1.1)',
      306 => 'Switch Proxy',
      307 => 'Temporary Redirect (since HTTP/1.1)',
      308 => 'Permanent Redirect (RFC 7538)',
      400 => 'Bad Request',
      401 => 'Unauthorized (RFC 7235)',
      402 => 'Payment Required',
      403 => 'Forbidden',
      404 => 'Not Found',
      405 => 'Method Not Allowed',
      406 => 'Not Acceptable',
      407 => 'Proxy Authentication Required (RFC 7235)',
      408 => 'Request Timeout',
      409 => 'Conflict',
      410 => 'Gone',
      411 => 'Length Required',
      412 => 'Precondition Failed (RFC 7232)',
      413 => 'Payload Too Large (RFC 7231)',
      414 => 'URI Too Long (RFC 7231)',
      415 => 'Unsupported Media Type',
      416 => 'Range Not Satisfiable (RFC 7233)',
      417 => 'Expectation Failed',
      418 => 'I\'m a teapot (RFC 2324, RFC 7168)',
      421 => 'Misdirected Request (RFC 7540)',
      422 => 'Unprocessable Entity (WebDAV; RFC 4918)',
      423 => 'Locked (WebDAV; RFC 4918)',
      424 => 'Failed Dependency (WebDAV; RFC 4918)',
      426 => 'Upgrade Required',
      428 => 'Precondition Required (RFC 6585)',
      429 => 'Too Many Requests (RFC 6585)',
      431 => 'Request Header Fields Too Large (RFC 6585)',
      451 => 'Unavailable For Legal Reasons (RFC 7725)',
      500 => 'Internal Server Error',
      501 => 'Not Implemented',
      502 => 'Bad Gateway',
      503 => 'Service Unavailable',
      504 => 'Gateway Timeout',
      505 => 'HTTP Version Not Supported',
      506 => 'Variant Also Negotiates (RFC 2295)',
      507 => 'Insufficient Storage (WebDAV; RFC 4918)',
      508 => 'Loop Detected (WebDAV; RFC 5842)',
      510 => 'Not Extended (RFC 2774)',
      511 => 'Network Authentication Required (RFC 6585)',
    ];
    if (!$message) {
      $message = isset($defaut_messages[$code]) ? $defaut_messages[$code] : 'No message';
    }
    $reponse = new Response();
    $reponse->setStatusCode($code, $message);
    $reponse->send();
  }

}
