<?php

namespace Drupal\spiro_utils\Services;

use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Node helper class
 */
class Http
{

  /**
   * Read JSON from HTTP response and return array
   * @param string $url
   * @param array $options Optional
   * @return array|null
   */
  public static function getJSON($url, array $options = [])
  {
    // Get response body
    $reponse = self::get($url, $options);
    if (!$reponse) {
      \Drupal::logger('spiro_utils_http')->error('No reponse for ' . $url);
      return null;
    }

    // Remove HTML comments that may be added to the reponse
    $reponse = preg_replace("/<!--(.*)-->/", '', $reponse);

    // Decode and return response
    return json_decode($reponse);
  }

  /**
   * Read and return HTTP response through Guzzle
   * @param string $url
   * @param array $options Ootional
   * @return string
   */
  public static function get($url, array $options = [])
  {
    // Get Guzzle client
    $client = \Drupal::httpClient();
    try {
      $response = $client->request('GET', $url, $options);
    } catch (GuzzleException $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    } catch (Exception $e) {
      \Drupal::logger('isimo_server')->error($e->getMessage(), ['exception' => $e]);
      return '';
    }

    // Check status
    if ($response->getStatusCode() !== 200) {
      \Drupal::logger('spiro_utils_http')->error($url . ' returned status ' . $response->getStatusCode());
    }

    // Return response body
    return (string)$response->getBody();
  }

  /**
   * Redirect (exists application)
   * @param string|NodeInterface $url string or node
   * @param array $parameters Optional (used when $url is a route)
   */
  public static function redirect($url, $parameters = [])
  {
    $response = new RedirectResponse(self::url($url, $parameters));
    $response->send();
  }

  /**
   * Render URL
   * @param string|NodeInterface $url string or node or route
   * @param array $parameters Optional (used when $url is a route)
   * @return string
   */
  public static function url($url, $parameters = [])
  {
    try {
      if ($url instanceof NodeInterface) {
        $url = $url->toUrl()->toString();
      } else if (strpos($url, '.')) {
        $url = Url::fromRoute($url, $parameters)->toString() ?: $url;
      }
    } catch (EntityMalformedException $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    } catch (Exception $e) {
      \Drupal::logger('spiro_utils_http')->error($e);
    }
    $url = (string)$url;
    if (strpos($url, '://') === false) {
      $url = \Drupal::request()->getSchemeAndHttpHost() . $url;
    }
    return $url;
  }

  /**
   * Send status code as response along with a message
   * @param int $code
   * @param null|string $message If message is omitted the standard message is used
   */
  public function sendStatus($code, $message = null)
  {
    $defaut_messages = [
      100 => 'Continue',
      101 => 'Switching Protocols',
      102 => 'Processing (WebDAV; RFC 2518)',
      103 => 'Early Hints (RFC 8297)',
      200 => 'OK',
      201 => 'Created',
      202 => 'Accepted',
      203 => 'Non-Authoritative Information (since HTTP/1.1)',
      204 => 'No Content',
      205 => 'Reset Content',
      206 => 'Partial Content (RFC 7233)',
      207 => 'Multi-Status (WebDAV; RFC 4918)',
      208 => 'Already Reported (WebDAV; RFC 5842)',
      226 => 'IM Used (RFC 3229)',
      300 => 'Multiple Choices',
      301 => 'Moved Permanently',
      302 => 'Found',
      303 => 'See Other (since HTTP/1.1)',
      304 => 'Not Modified (RFC 7232)',
      305 => 'Use Proxy (since HTTP/1.1)',
      306 => 'Switch Proxy',
      307 => 'Temporary Redirect (since HTTP/1.1)',
      308 => 'Permanent Redirect (RFC 7538)',
      400 => 'Bad Request',
      401 => 'Unauthorized (RFC 7235)',
      402 => 'Payment Required',
      403 => 'Forbidden',
      404 => 'Not Found',
      405 => 'Method Not Allowed',
      406 => 'Not Acceptable',
      407 => 'Proxy Authentication Required (RFC 7235)',
      408 => 'Request Timeout',
      409 => 'Conflict',
      410 => 'Gone',
      411 => 'Length Required',
      412 => 'Precondition Failed (RFC 7232)',
      413 => 'Payload Too Large (RFC 7231)',
      414 => 'URI Too Long (RFC 7231)',
      415 => 'Unsupported Media Type',
      416 => 'Range Not Satisfiable (RFC 7233)',
      417 => 'Expectation Failed',
      418 => 'I\'m a teapot (RFC 2324, RFC 7168)',
      421 => 'Misdirected Request (RFC 7540)',
      422 => 'Unprocessable Entity (WebDAV; RFC 4918)',
      423 => 'Locked (WebDAV; RFC 4918)',
      424 => 'Failed Dependency (WebDAV; RFC 4918)',
      426 => 'Upgrade Required',
      428 => 'Precondition Required (RFC 6585)',
      429 => 'Too Many Requests (RFC 6585)',
      431 => 'Request Header Fields Too Large (RFC 6585)',
      451 => 'Unavailable For Legal Reasons (RFC 7725)',
      500 => 'Internal Server Error',
      501 => 'Not Implemented',
      502 => 'Bad Gateway',
      503 => 'Service Unavailable',
      504 => 'Gateway Timeout',
      505 => 'HTTP Version Not Supported',
      506 => 'Variant Also Negotiates (RFC 2295)',
      507 => 'Insufficient Storage (WebDAV; RFC 4918)',
      508 => 'Loop Detected (WebDAV; RFC 5842)',
      510 => 'Not Extended (RFC 2774)',
      511 => 'Network Authentication Required (RFC 6585)',
    ];
    if (!$message) {
      $message = isset($defaut_messages[$code]) ? $defaut_messages[$code] : 'No message';
    }
    $reponse = new Response();
    $reponse->setStatusCode($code, $message);
    $reponse->send();
  }

  /**
   * Get the content of an url, including header and cert
   *
   * @param string $url
   * @param mixed[] $stream_options
   *
   * @return PHPDOC_GetWithCert|object
   * @throws Exception
   */
  public static function getWithCert($url, $stream_options = NULL)
  {
    $ssl = (substr($url, 0, 7) !== 'http://');
    $url_parts = parse_url($url) + ['port' => $ssl ? 443 : 80, 'path' => '/', 'query' => NULL, 'user' => NULL, 'pass' => NULL];
    $tcp_url = 'tcp://' . $url_parts['host'] . ':' . $url_parts['port'];
    $headers = [
      'Host: ' . $url_parts['host']
    ];
    if($url_parts['user']) {
      $headers[] = 'Authorization: Basic ' . base64_encode($url_parts['user'] . ':' . $url_parts['pass']);
    }
    $http_request = implode(
      "\r\n",
      [
        'GET ' . $url_parts['path'] . ($url_parts['query'] ? '?' . $url_parts['query'] : '') . ' HTTP/1.1',
        implode("\r\n", $headers),
        'Connection: close',
        '', // end of head
        '', // end of body
      ]
    );

    if(!is_array($stream_options))
    {
      $stream_options = [];
    }
    $stream_options['ssl']['capture_peer_cert'] = TRUE;
    //$stream_options['ssl']['capture_peer_cert_chain'] = TRUE;
    $context = stream_context_create($stream_options);
    $cert = '';

    $client = stream_socket_client($tcp_url, $errno, $errstr, 30, STREAM_CLIENT_CONNECT, $context);

    if($ssl)
    {
      stream_socket_enable_crypto($client, TRUE, STREAM_CRYPTO_METHOD_SSLv23_CLIENT);
      $info = stream_context_get_params($client);
      openssl_x509_export($info['options']['ssl']['peer_certificate'], $cert);
    }

    fwrite($client, $http_request);
    $data = stream_get_contents($client);
    fclose($client);

    [$head, $body] = explode("\r\n\r\n", $data, 2) + ['', ''];
    $head_rows = explode("\n", $head);
    // Merge split headers
    foreach(range(count($head_rows) - 1, 1, -1) as $position)
    {
      if(!trim($head_rows[$position][0]))
      {
        $head_rows[$position - 1][0] = rtrim($head_rows[$position - 1][0]) . ' ' . trim($head_rows[$position][0]);
      }
    }
    $status_row = array_shift($head_rows);
    $status_parts = explode(' ', $status_row, 3);
    $status = (int) $status_parts[1];
    $status_text = trim($status_parts[2]);

    $headers = [];
    foreach($head_rows as $row)
    {
      [$key, $value] = explode(': ', $row);
      $headers[$key][] = $value;
    }

    if(isset($headers['Transfer-Encoding'][0]))
    {
      foreach(explode(',', $headers['Transfer-Encoding'][0]) as $encoding)
      {
        switch(trim(strtolower($encoding)))
        {
          case 'chunked':
            $raw = $body;
            $body = '';
            while($raw)
            {
              if(!preg_match("#^\r?\n?([0-9a-f-A-F]+)\r?\n#", $raw, $hits))
              {
                throw new \Exception('Parsing Transfer-Encoding ' . $encoding . ' failed on');
              }
              $offset = strlen($hits[0]);
              $hex = $hits[1];
              $length = hexdec($hex);
              if(!$length AND $hex === '0')
              {
                unset($raw);
                break;
              }
              $body .= substr($raw, $offset, $length);
              $raw = substr($raw, $offset + $length);
            }
            break;

          case 'compress':
          case 'deflate':
          case 'gzip':
          case 'identity':
          default:
            throw new \Exception('Non implemented transfer encoding: ' . $encoding);
        }
      }
    }

    return (object) compact(
      'ssl',
      'url',
      'status',
      'status_text',
      'head',
      'headers',
      //'data',
      'body',
      'cert'
    );
  }

  /**
   * Get json responce, and also include ssl-certificate
   *
   * @param string $url
   * @param string $cert_key
   * @param string $wrap_key
   * @param mixed[] $stream_options
   *
   * @return string[]|mixed[]|null
   * @throws Exception
   */
  public static function getJSONWithCert($url, $cert_key = 'cert', $wrap_key = NULL, $stream_options = NULL)
  {
    // Get response body
    $reponse = self::getWithCert($url, $stream_options);
    if(!$reponse)
    {
      \Drupal::logger('spiro_utils_http')->error('No reponse for ' . $url);
      return NULL;
    }
    if($reponse->status !== 200)
    {
      \Drupal::logger('spiro_utils_http')->error('Bad reponse ' . $reponse->status . ' for ' . $url);
      return NULL;
    }

    // Remove HTML comments that may be added to the reponse
    $json_raw = preg_replace("/<!--(.*)-->/", '', $reponse->body);

    // Decode response
    $json = json_decode($json_raw);

    if(!$json)
    {
      return $json;
    }

    if($wrap_key)
    {
      return [
        $wrap_key => $json,
        $cert_key => $reponse->cert,
      ];
    }

    $json->$cert_key = $reponse->cert;
    return $json;
  }

}
